/*
 * ^HdI2C1.c
 *
 *  Created on: 10 ago 2019
 *      Author: andre
 */

#include "^Common.h"

#define HD44780_ADDR 0x27   //+++ Define the slave address of device on the LCD bus.
#define Bl 0b00001000       //+++ Backlight enable bit (On = 1, Off =0).
#define En 0b00000100       //+++ Enable bit (Enable on low edge).
#define Rw 0b00000010       //+++ Read/Write bit (Read = 1, Write = 0).
#define Rs 0b00000001       //+++ Register select bit (Data = 1, Control = 0).

//1-line display (N = 0)
//2-line display (N = 1)

//F = 1: 5  10 dots, F = 0: 5  8 dots

//I/D = 1: Increment
//I/D = 0: Decrement

//S: Shifts the entire display either to the right (I/D = 0) or to the left (I/D = 1) when S is 1. The display does
//not shift if S is 0.
//If S is 1, it will seem as if the cursor does not move but the display does. The display does not shift when
//reading from DDRAM. Also, writing into or reading out from CGRAM does not shift the display.

//DL = 1: 8 bits, DL = 0: 4 bits

//Function
//set
//0 0 0 0 1 DL N F ? ? Sets interface data length
//(DL), number of display lines
//(N), and character font (F).

//Busy Flag (BF)
//When the busy flag is 1, the HD44780U is in the internal operation mode, and the next instruction will not
//be accepted. When RS = 0 and R/W = 1 (Table 1), the busy flag is output to DB7. The next instruction
//must be written after ensuring that the busy flag is 0.

//AC: Address counter used for both DD and CGRAM addresses.

#pragma DATA_ALIGN(I2cTaskStack, 8)  //+++ 1-byte alignment on the stack to avoid wasting memory.
static uint8_t I2cTaskStack[STACKSIZE];
Task_Struct I2cTr;
I2C_Handle hI2C;
I2C_Transaction stI2cTransact;
uint8_t aucTxBuf[4];
uint8_t ucFirstRow = 0x80, ucSecondRow = 0xC0;

Void I2cTf(UArg arg0, UArg arg1);
void LcdInit(void);
void LcdSendCmd(uint8_t ucInfo, uint32_t uiWaitMicroSec);   //+++ Send a command to display.
void LcdSendData(uint8_t ucInfo, uint32_t uiWaitMicroSec);  //+++ Ask display for a symbol.
void LcdSendString(uint8_t *pucString, uint32_t uiWaitMicroSec); //+++ Send string to LCD.
void Error(void);

void StartI2cTf(void)
{
    Task_Params taskParams;
    Task_Params_init(&taskParams);
    taskParams.stackSize = STACKSIZE;
    taskParams.priority = 2;    //+++ More than 1 or is preempted by while (1) in mainThread.
    taskParams.stack = &I2cTaskStack;

    Task_construct(&I2cTr, I2cTf, &taskParams, NULL);
}

Void I2cTf(UArg arg0, UArg arg1)
{
    I2C_Params      i2cParams;

    /* Create I2C for usage */
    I2C_Params_init(&i2cParams);
    i2cParams.bitRate = I2C_100kHz;
    i2cParams.transferMode = I2C_MODE_BLOCKING;
    hI2C = I2C_open(Board_I2C_TMP, &i2cParams);
    if (hI2C == NULL)
        Error();

    /* Common I2C transaction setup */
    stI2cTransact.writeBuf   = aucTxBuf;
    stI2cTransact.writeCount = sizeof(aucTxBuf);
    stI2cTransact.readBuf    = NULL;
    stI2cTransact.readCount  = 0;
    stI2cTransact.slaveAddress = HD44780_ADDR;  //+++ CC1310 prepends this byte for each transaction.

    sleep(2);   //+++ Necessary to allow HD44780 LCD 1602 + backpack PCF 8574 to stabilize.
    PIN_setOutputValue(hDynPin, TXS0108E_OE, 1);    //+++ Connect I2C bus between display and CC1310.

    LcdInit();    //+++
    PIN_setOutputValue(hDynPin, RED_LED, 0);
    PIN_setOutputValue(hDynPin, GREEN_LED, 1);

    LcdSendCmd(0x01, 100); //+++ Clear display.
    uint8_t ucCount;
    for (ucCount = 0; ucCount < 100; ucCount++)
    {
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucFirstRow, 100);
        LcdSendString("Impisa tuto e                           ", 100);
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucSecondRow, 100);
        LcdSendString("speta un per de                         ", 100);
        sleep(3);
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucFirstRow, 100);
        LcdSendString("secondi. Quando                         ", 100);
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucSecondRow, 100);
        LcdSendString("che te stui,                            ", 100);
        sleep(3);
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucFirstRow, 100);
        LcdSendString("speta un fia de                         ", 100);
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucSecondRow, 100);
        LcdSendString("secondi che se                          ", 100);
        sleep(3);
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucFirstRow, 100);
        LcdSendString("scarichi tuti i                         ", 100);
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucSecondRow, 100);
        LcdSendString("circuiti. Dopo                          ", 100);
        sleep(3);
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucFirstRow, 100);
        LcdSendString("te tachi de                             ", 100);
        LcdSendCmd(0x02, 2000); //+++ Return home.
        LcdSendCmd(ucSecondRow, 100);
        LcdSendString("novo. Ci vidin!                         ", 100);
        sleep(3);
    }

    I2C_close(hI2C);
}

//+++ Initialise LCD Figure 24 4-Bit Interface of HD44780U (LCD-II), (Dot Matrix Liquid Crystal Display Controller_Driver)
void LcdInit(void)
{
    LcdSendCmd(0x30, 50000);    //+++ LCD_8BIT_INIT.
    LcdSendCmd(0x30, 2000);     //+++ LCD_8BIT_INIT.
    LcdSendCmd(0x30, 2000);     //+++ LCD_8BIT_INIT.
    LcdSendCmd(0x20, 2000);     //+++ LCD_4BIT_INIT.
    LcdSendCmd(0x20, 2000);     //+++ LCD_4BIT_INIT.
    LcdSendCmd(0x80, 2000);     //+++ Specify the number of display lines and character font.
    LcdSendCmd(0x00, 2000);     //
    LcdSendCmd(0x80, 2000);     //
    LcdSendCmd(0x00, 2000);     //
    LcdSendCmd(0x01, 2000);     //+++ Clear display.
    LcdSendCmd(0x00, 2000);     //
    LcdSendCmd(0x60, 2000);     //+++ Entry mode set.
}

void LcdSendCmd(uint8_t ucInfo, uint32_t uiWaitMicroSec)
{
    uint8_t data_l, data_u;
    data_l = (ucInfo << 4) & 0xF0;  //+++ Select lower nibble by moving it to the upper nibble position.
    data_u = ucInfo & 0xF0;  //+++ Select upper nibble.

    aucTxBuf[0] = data_u | 0x0C;   //+++ enable = 1 and rs = 0.
    aucTxBuf[1] = data_u | 0x08;   //+++ enable = 0 and rs = 0.
    aucTxBuf[2] = data_l | 0x0C;   //+++ enable = 1 and rs = 0.
    aucTxBuf[3] = data_l | 0x08;   //+++ enable = 0 and rs = 0.

    if (!I2C_transfer(hI2C, &stI2cTransact))
        Error();

    usleep(uiWaitMicroSec);
}

void LcdSendData(uint8_t ucInfo, uint32_t uiWaitMicroSec)
{
    uint8_t data_l, data_u;
    data_l = (ucInfo << 4) & 0xF0;  //+++ Select lower nibble by moving it to the upper nibble position.
    data_u = ucInfo & 0xF0;  //+++ Select upper nibble.

    aucTxBuf[0] = data_u | 0x0D;   //+++ enable = 1 and rs = 1.
    aucTxBuf[1] = data_u | 0x09;   //+++ enable = 0 and rs = 1.
    aucTxBuf[2] = data_l | 0x0D;   //+++ enable = 1 and rs = 1.
    aucTxBuf[3] = data_l | 0x09;   //+++ enable = 0 and rs = 1.

    if (!I2C_transfer(hI2C, &stI2cTransact))
        Error();

    usleep(uiWaitMicroSec);
}

void LcdSendString(uint8_t *pucString, uint32_t uiWaitMicroSec)
{
    while (*pucString)
        LcdSendData(*pucString++, uiWaitMicroSec);
}

void Error(void)
{
    PIN_setOutputValue(hDynPin, RED_LED, 1);
    PIN_setOutputValue(hDynPin, GREEN_LED, 0);
    while (1);
}
